const jwt = require('jsonwebtoken');
const rateLimit = require('express-rate-limit');
const config = require('../config/config');
const database = require('../models/database');

// ============================================================================
// RATE LIMITING MIDDLEWARE
// ============================================================================

/**
 * General rate limiting for all endpoints
 */
const generalRateLimit = rateLimit({
  windowMs: config.RATE_LIMIT_WINDOW_MS,
  max: config.RATE_LIMIT_MAX,
  message: {
    error: 'RATE_LIMIT_EXCEEDED',
    message: 'Too many requests from this IP, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  skip: (req, res) => {
    // Skip rate limiting for health checks
    return req.path === '/health' || req.path === '/status';
  }
});

/**
 * Strict rate limiting for authentication endpoints
 */
const strictAuthRateLimit = rateLimit({
  windowMs: config.RATE_LIMIT_WINDOW_MS,
  max: config.AUTH_RATE_LIMIT_MAX,
  message: {
    error: 'AUTH_RATE_LIMIT_EXCEEDED',
    message: 'Too many authentication attempts, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  skipSuccessfulRequests: true, // Don't count successful requests
});

// ============================================================================
// JWT AUTHENTICATION MIDDLEWARE
// ============================================================================

/**
 * Verify JWT token from Cookie or Authorization header
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 * @param {function} next - Express next function
 */
function verifyToken(req, res, next) {
  try {
    // Try to get token from cookie first, then from Authorization header
    const token = req.cookies.auth_token || 
      (req.headers.authorization && req.headers.authorization.startsWith('Bearer ') ? 
        req.headers.authorization.slice(7) : req.headers.authorization);
    
    if (!token) {
      return res.status(401).json({
        message: 'Access token required',
        error: 'NO_TOKEN'
      });
    }
    
    // Verify JWT token
    jwt.verify(token, config.JWT_SECRET, async (err, decoded) => {
      if (err) {
        console.error('❌ JWT verification error:', err.message);
        
        if (err.name === 'TokenExpiredError') {
          return res.status(401).json({
            message: 'Token expired',
            error: 'TOKEN_EXPIRED'
          });
        } else if (err.name === 'JsonWebTokenError') {
          return res.status(401).json({
            message: 'Invalid token',
            error: 'INVALID_TOKEN'
          });
        } else {
          return res.status(401).json({
            message: 'Token verification failed',
            error: 'TOKEN_VERIFICATION_FAILED'
          });
        }
      }
      
      try {
        // Verify user still exists in database
        const user = await database.get(
          'SELECT id, email, crm_access, active FROM users WHERE id = ?',
          [decoded.id]
        );
        
        if (!user) {
          return res.status(401).json({
            message: 'User not found',
            error: 'USER_NOT_FOUND'
          });
        }
        
        if (!user.active) {
          return res.status(401).json({
            message: 'User account is disabled',
            error: 'USER_DISABLED'
          });
        }
        
        // Add user info to request object
        req.user = {
          id: user.id,
          email: user.email,
          crm_access: !!user.crm_access
        };
        
        // Update last activity (optional)
        if (config.isFeatureEnabled('TRACK_USER_ACTIVITY')) {
          database.run(
            'UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = ?',
            [user.id]
          ).catch(err => {
            console.error('❌ Error updating last activity:', err);
            // Don't fail the request for this
          });
        }
        
        next();
        
      } catch (dbError) {
        console.error('❌ Database error during token verification:', dbError);
        res.status(500).json({
          message: 'Internal server error during authentication',
          error: 'AUTH_DATABASE_ERROR'
        });
      }
    });
    
  } catch (error) {
    console.error('❌ Authentication middleware error:', error);
    res.status(500).json({
      message: 'Internal authentication error',
      error: 'AUTH_INTERNAL_ERROR'
    });
  }
}

/**
 * Optional authentication - doesn't fail if no token provided
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 * @param {function} next - Express next function
 */
function optionalAuth(req, res, next) {
  // Try to get token from cookie first, then from Authorization header
  const token = req.cookies.auth_token || 
    (req.headers.authorization && req.headers.authorization.startsWith('Bearer ') ? 
      req.headers.authorization.slice(7) : req.headers.authorization);
  
  if (!token) {
    // No token provided, continue without authentication
    req.user = null;
    return next();
  }
  
  // Token provided, verify it
  verifyToken(req, res, next);
}

// ============================================================================
// PERMISSION MIDDLEWARE
// ============================================================================

/**
 * Require CRM access permissions
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 * @param {function} next - Express next function
 */
function requireCrmAccess(req, res, next) {
  if (!req.user) {
    return res.status(401).json({
      message: 'Authentication required',
      error: 'AUTHENTICATION_REQUIRED'
    });
  }
  
  if (!req.user.crm_access) {
    return res.status(403).json({
      message: 'CRM access required',
      error: 'CRM_ACCESS_DENIED'
    });
  }
  
  next();
}

/**
 * Require admin permissions (placeholder for future implementation)
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 * @param {function} next - Express next function
 */
function requireAdmin(req, res, next) {
  if (!req.user) {
    return res.status(401).json({
      message: 'Authentication required',
      error: 'AUTHENTICATION_REQUIRED'
    });
  }
  
  // TODO: Implement admin role check
  // For now, assume CRM access users are admins
  if (!req.user.crm_access) {
    return res.status(403).json({
      message: 'Admin access required',
      error: 'ADMIN_ACCESS_DENIED'
    });
  }
  
  next();
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

/**
 * Generate JWT token for user
 * @param {object} user - User object with id and email
 * @param {string} expiresIn - Token expiration time
 * @returns {string} JWT token
 */
function generateToken(user, expiresIn = config.JWT_EXPIRES_IN) {
  const payload = {
    id: user.id,
    email: user.email,
    iat: Math.floor(Date.now() / 1000)
  };
  
  return jwt.sign(payload, config.JWT_SECRET, { expiresIn });
}

/**
 * Decode JWT token without verification (for debugging)
 * @param {string} token - JWT token
 * @returns {object|null} Decoded token payload or null
 */
function decodeToken(token) {
  try {
    return jwt.decode(token);
  } catch (error) {
    console.error('❌ Token decode error:', error);
    return null;
  }
}

/**
 * Check if token is expired
 * @param {string} token - JWT token
 * @returns {boolean} True if token is expired
 */
function isTokenExpired(token) {
  try {
    const decoded = jwt.decode(token);
    if (!decoded || !decoded.exp) {
      return true;
    }
    
    const currentTime = Math.floor(Date.now() / 1000);
    return decoded.exp < currentTime;
  } catch (error) {
    return true;
  }
}

// ============================================================================
// INPUT VALIDATION MIDDLEWARE
// ============================================================================

/**
 * Validate required fields in request body
 * @param {Array} requiredFields - Array of required field names
 * @returns {function} Express middleware function
 */
function validateRequiredFields(requiredFields) {
  return (req, res, next) => {
    const missingFields = [];
    
    for (const field of requiredFields) {
      if (!req.body[field] || req.body[field].toString().trim() === '') {
        missingFields.push(field);
      }
    }
    
    if (missingFields.length > 0) {
      return res.status(400).json({
        message: 'Missing required fields',
        error: 'MISSING_REQUIRED_FIELDS',
        missing_fields: missingFields
      });
    }
    
    next();
  };
}

/**
 * Validate email format
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 * @param {function} next - Express next function
 */
function validateEmail(req, res, next) {
  const { email } = req.body;
  
  if (!email) {
    return next(); // Let required field validation handle this
  }
  
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  
  if (!emailRegex.test(email)) {
    return res.status(400).json({
      message: 'Invalid email format',
      error: 'INVALID_EMAIL_FORMAT'
    });
  }
  
  next();
}

/**
 * Sanitize input fields
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 * @param {function} next - Express next function
 */
function sanitizeInput(req, res, next) {
  // Trim whitespace from string fields
  if (req.body && typeof req.body === 'object') {
    for (const key in req.body) {
      if (typeof req.body[key] === 'string') {
        req.body[key] = req.body[key].trim();
      }
    }
  }
  
  next();
}

// ============================================================================
// EXPORTS
// ============================================================================

module.exports = {
  // Rate limiting
  generalRateLimit,
  strictAuthRateLimit,
  
  // Authentication
  verifyToken,
  optionalAuth,
  
  // Permissions
  requireCrmAccess,
  requireAdmin,
  
  // Utilities
  generateToken,
  decodeToken,
  isTokenExpired,
  
  // Validation
  validateRequiredFields,
  validateEmail,
  sanitizeInput
};