const sqlite3 = require('sqlite3').verbose();
const config = require('../config/config');
const fs = require('fs');
const path = require('path');

let db = null;

/**
 * Initialize database connection
 */
function initialize() {
  return new Promise((resolve, reject) => {
    const dbPath = config.DB_PATH || path.join(__dirname, '../../ayurwell_platform.sqlite');
    console.log(`📍 Connecting to database: ${dbPath}`);
    
    const dbDir = path.dirname(dbPath);
    if (!fs.existsSync(dbDir)) {
      fs.mkdirSync(dbDir, { recursive: true });
    }
    
    db = new sqlite3.Database(dbPath, (err) => {
      if (err) {
        console.error('❌ Database connection error:', err.message);
        reject(err);
      } else {
        console.log('✅ Connected to SQLite database');
        
        db.serialize(() => {
          db.run('PRAGMA foreign_keys = ON');
          db.run('PRAGMA journal_mode = WAL');
          db.run('PRAGMA synchronous = NORMAL');
        });
        
        resolve();
      }
    });
  });
}

/**
 * Get database instance
 */
function getDatabase() {
  if (!db) {
    throw new Error('Database not initialized. Call initialize() first.');
  }
  return db;
}

/**
 * Execute a query and return a single row
 */
function get(sql, params = []) {
  return new Promise((resolve, reject) => {
    if (!db) {
      reject(new Error('Database not initialized'));
      return;
    }
    
    db.get(sql, params, (err, row) => {
      if (err) {
        console.error('❌ Database GET error:', err);
        reject(err);
      } else {
        resolve(row || null);
      }
    });
  });
}

/**
 * Execute a query and return all rows
 */
function query(sql, params = []) {
  return new Promise((resolve, reject) => {
    if (!db) {
      reject(new Error('Database not initialized'));
      return;
    }
    
    db.all(sql, params, (err, rows) => {
      if (err) {
        console.error('❌ Database QUERY error:', err);
        reject(err);
      } else {
        resolve(rows || []);
      }
    });
  });
}

/**
 * Execute an INSERT, UPDATE, or DELETE query
 */
function run(sql, params = []) {
  return new Promise((resolve, reject) => {
    if (!db) {
      reject(new Error('Database not initialized'));
      return;
    }
    
    db.run(sql, params, function(err) {
      if (err) {
        console.error('❌ Database RUN error:', err);
        reject(err);
      } else {
        resolve({
          lastID: this.lastID,
          changes: this.changes
        });
      }
    });
  });
}

/**
 * Close database connection
 */
function close() {
  return new Promise((resolve, reject) => {
    if (db) {
      db.close((err) => {
        if (err) {
          console.error('❌ Error closing database:', err);
          reject(err);
        } else {
          console.log('✅ Database connection closed');
          db = null;
          resolve();
        }
      });
    } else {
      resolve();
    }
  });
}

module.exports = {
  initialize,
  getDatabase,
  get,
  query,
  run,
  close
};