const express = require('express');
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const database = require('../models/database');
const config = require('../config/config');
const {
  strictAuthRateLimit,
  validateRequiredFields,
  validateEmail,
  sanitizeInput,
  generateToken
} = require('../middleware/auth');

const router = express.Router();

// ============================================================================
// USER REGISTRATION
// ============================================================================

/**
 * POST /register
 * Register a new user account
 */
router.post('/register', 
  strictAuthRateLimit,
  sanitizeInput,
  validateRequiredFields(['email', 'password']),
  validateEmail,
  async (req, res) => {
    try {
      const { email, password, first_name, last_name } = req.body;

      console.log('🔐 Registration attempt for:', email);

      // Check if user already exists
      const existingUser = await database.get(
        'SELECT id FROM users WHERE email = ?',
        [email.toLowerCase()]
      );

      if (existingUser) {
        console.log('❌ Registration failed: Email already exists');
        return res.status(409).json({
          message: 'Email address already registered',
          error: 'EMAIL_ALREADY_EXISTS'
        });
      }

      // Validate password strength (basic validation)
      if (password.length < 6) {
        return res.status(400).json({
          message: 'Password must be at least 6 characters long',
          error: 'PASSWORD_TOO_SHORT'
        });
      }

      // Hash password
      const hashedPassword = await bcrypt.hash(password, config.BCRYPT_SALT_ROUNDS);

      // Create user record
      const result = await database.run(`
        INSERT INTO users (email, password, first_name, last_name, crm_access)
        VALUES (?, ?, ?, ?, ?)
      `, [
        email.toLowerCase(),
        hashedPassword,
        first_name || null,
        last_name || null,
        0 // No CRM access by default - admin grants manually
      ]);

      if (!result.lastID) {
        throw new Error('Failed to create user record');
      }

      // Generate JWT token
      const token = generateToken({
        id: result.lastID,
        email: email.toLowerCase()
      });

      console.log('✅ User registered successfully:', email);

      // Set HTTP-only cookie and return user data
      res.cookie('auth_token', token, {
        httpOnly: true,
        secure: config.IS_PRODUCTION,
        sameSite: 'strict',
        maxAge: 7 * 24 * 60 * 60 * 1000 // 7 days
      }).status(201).json({
        message: 'Registration successful',
        user: {
          id: result.lastID,
          email: email.toLowerCase(),
          first_name: first_name || null,
          last_name: last_name || null,
          crm_access: false
        }
      });

    } catch (error) {
      console.error('❌ Registration error:', error);
      res.status(500).json({
        message: 'Registration failed',
        error: 'REGISTRATION_ERROR'
      });
    }
  }
);

// ============================================================================
// USER LOGIN
// ============================================================================

/**
 * POST /login
 * Authenticate user and return JWT token
 */
router.post('/login',
  strictAuthRateLimit,
  sanitizeInput,
  validateRequiredFields(['email', 'password']),
  validateEmail,
  async (req, res) => {
    try {
      const { email, password } = req.body;

      console.log('🔐 Login attempt for:', email);

      // Find user by email
      const user = await database.get(
        'SELECT id, email, password, crm_access, first_name, last_name, active FROM users WHERE email = ?',
        [email.toLowerCase()]
      );

      if (!user) {
        console.log('❌ Login failed: User not found');
        return res.status(401).json({
          message: 'Invalid email or password',
          error: 'INVALID_CREDENTIALS'
        });
      }

      // Check if user account is active
      if (!user.active) {
        console.log('❌ Login failed: User account disabled');
        return res.status(401).json({
          message: 'Account is disabled',
          error: 'ACCOUNT_DISABLED'
        });
      }

      // Verify password
      const isPasswordValid = await bcrypt.compare(password, user.password);

      if (!isPasswordValid) {
        console.log('❌ Login failed: Invalid password');
        return res.status(401).json({
          message: 'Invalid email or password',
          error: 'INVALID_CREDENTIALS'
        });
      }

      // Update last login timestamp
      database.run(
        'UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = ?',
        [user.id]
      ).catch(err => {
        console.error('❌ Error updating last login:', err);
        // Don't fail the login for this
      });

      // Generate JWT token
      const token = generateToken({
        id: user.id,
        email: user.email
      });

      console.log('✅ Login successful for:', email);

      // Set HTTP-only cookie and return user data
      res.cookie('auth_token', token, {
        httpOnly: true,
        secure: config.IS_PRODUCTION,
        sameSite: 'strict',
        maxAge: 7 * 24 * 60 * 60 * 1000 // 7 days
      }).json({
        message: 'Login successful',
        user: {
          id: user.id,
          email: user.email,
          first_name: user.first_name,
          last_name: user.last_name,
          crm_access: !!user.crm_access
        }
      });

    } catch (error) {
      console.error('❌ Login error:', error);
      res.status(500).json({
        message: 'Login failed',
        error: 'LOGIN_ERROR'
      });
    }
  }
);

// ============================================================================
// TOKEN VERIFICATION
// ============================================================================

/**
 * GET /verify-token
 * Verify if the provided JWT token is valid
 */
router.get('/verify-token', async (req, res) => {
  try {
    // Try to get token from cookie first, then from header
    const token = req.cookies.auth_token || 
      (req.headers.authorization && req.headers.authorization.startsWith('Bearer ') ? 
        req.headers.authorization.slice(7) : req.headers.authorization);
    
    if (!token) {
      return res.json({ 
        valid: false, 
        error: 'NO_TOKEN' 
      });
    }
    
    // Verify JWT token
    jwt.verify(token, config.JWT_SECRET, async (err, decoded) => {
      if (err) {
        console.log('❌ Token verification failed:', err.message);
        return res.json({ 
          valid: false, 
          error: err.name === 'TokenExpiredError' ? 'TOKEN_EXPIRED' : 'INVALID_TOKEN'
        });
      }
      
      try {
        // Check if user still exists and is active
        const user = await database.get(
          'SELECT id, email, crm_access, active FROM users WHERE id = ?',
          [decoded.id]
        );

        if (!user) {
          return res.json({ 
            valid: false, 
            error: 'USER_NOT_FOUND' 
          });
        }
        
        if (!user.active) {
          return res.json({ 
            valid: false, 
            error: 'USER_DISABLED' 
          });
        }

        res.json({ 
          valid: true,
          user: {
            id: user.id,
            email: user.email,
            crm_access: !!user.crm_access
          }
        });

      } catch (dbError) {
        console.error('❌ Token verification database error:', dbError);
        res.json({ 
          valid: false, 
          error: 'DATABASE_ERROR' 
        });
      }
    });

  } catch (error) {
    console.error('❌ Token verification error:', error);
    res.json({ 
      valid: false, 
      error: 'VERIFICATION_ERROR' 
    });
  }
});

// ============================================================================
// LOGOUT ENDPOINT
// ============================================================================

/**
 * POST /logout
 * Logout user (clear cookie)
 */
router.post('/logout', (req, res) => {
  try {
    console.log('🚪 User logout');
    
    // Clear the auth cookie
    res.clearCookie('auth_token', {
      httpOnly: true,
      secure: config.IS_PRODUCTION,
      sameSite: 'strict'
    }).json({ 
      message: 'Logout successful'
    });

  } catch (error) {
    console.error('❌ Logout error:', error);
    res.status(500).json({ 
      message: 'Logout failed',
      error: 'LOGOUT_ERROR'
    });
  }
});

// ============================================================================
// PASSWORD RESET ENDPOINTS (Future Implementation)
// ============================================================================

/**
 * POST /forgot-password
 * Request password reset (placeholder for future implementation)
 */
router.post('/forgot-password', strictAuthRateLimit, (req, res) => {
  res.status(501).json({ 
    message: 'Password reset not implemented yet',
    error: 'NOT_IMPLEMENTED'
  });
});

/**
 * POST /reset-password
 * Reset password with token (placeholder for future implementation)
 */
router.post('/reset-password', strictAuthRateLimit, (req, res) => {
  res.status(501).json({ 
    message: 'Password reset not implemented yet',
    error: 'NOT_IMPLEMENTED'
  });
});

// ============================================================================
// TEST ENDPOINT  
// ============================================================================

router.get('/test', (req, res) => {
  res.json({ message: 'Auth router works!' });
});

// TEST POST endpoint
router.post('/test-post', (req, res) => {
  res.json({ 
    message: 'POST works!', 
    body: req.body 
  });
});

// TEST registration without middleware
router.post('/test-register', async (req, res) => {
  try {
    const { email, password, first_name, last_name } = req.body;
    
    console.log('📦 Test register data:', { email, password, first_name, last_name });
    
    res.json({ 
      message: 'Test register works!', 
      received: { email, password, first_name, last_name }
    });
  } catch (error) {
    console.error('❌ Test register error:', error);
    res.status(500).json({ error: error.message });
  }
});

// TEST registration with sanitizeInput only
router.post('/test-register', 
  sanitizeInput,
  async (req, res) => {
    try {
      const { email, password, first_name, last_name } = req.body;
      res.json({ 
        message: 'Test with sanitizeInput works!', 
        received: { email, password, first_name, last_name }
      });
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  }
);

// ============================================================================
// EXPORTS
// ============================================================================

module.exports = router;