const express = require('express');
const bcrypt = require('bcrypt');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const database = require('../models/database');
const config = require('../config/config');
const { 
  verifyToken, 
  validateRequiredFields, 
  sanitizeInput 
} = require('../middleware/auth');

const router = express.Router();

// Apply authentication to all client routes
router.use(verifyToken);

// ============================================================================
// MULTER SETUP FOR AVATAR UPLOADS
// ============================================================================

// Ensure uploads directory exists
const uploadsDir = path.join(__dirname, '../../uploads/avatars');
if (!fs.existsSync(uploadsDir)) {
  fs.mkdirSync(uploadsDir, { recursive: true });
}

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadsDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'avatar-' + req.user.id + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 5 * 1024 * 1024 // 5MB
  },
  fileFilter: (req, file, cb) => {
    if (file.mimetype.startsWith('image/')) {
      cb(null, true);
    } else {
      cb(new Error('Only image files are allowed'), false);
    }
  }
});

// ============================================================================
// CLIENT PROFILE MANAGEMENT
// ============================================================================

/**
 * GET /api/client/profile
 * Get client profile information
 */
router.get('/profile', async (req, res) => {
  try {
    const userId = req.user.id;

    // Get user basic info
    const user = await database.get(
      'SELECT id, email, first_name, last_name, user_type, created_at FROM users WHERE id = ?',
      [userId]
    );

    if (!user) {
      return res.status(404).json({ 
        message: 'User not found',
        error: 'USER_NOT_FOUND'
      });
    }

    // Get extended profile info
    const profile = await database.get(
      'SELECT * FROM platform_clients WHERE user_id = ?',
      [userId]
    );

    // Get user settings
    const settings = await database.get(
      'SELECT language, timezone, notifications, privacy, display FROM platform_user_settings WHERE user_id = ?',
      [userId]
    );

    // Parse JSON settings
    let parsedSettings = {
      language: 'pl',
      timezone: 'Europe/Warsaw',
      notifications: {
        email_marketing: true,
        email_bookings: true,
        email_reminders: true,
        push_marketing: false,
        push_bookings: true,
        push_reminders: true
      },
      privacy: {
        profile_visibility: 'public',
        show_activity: false,
        allow_contact: true,
        data_analytics: true
      },
      display: {
        theme: 'system',
        compact_mode: false,
        animations: true
      }
    };

    if (settings) {
      parsedSettings.language = settings.language || 'pl';
      parsedSettings.timezone = settings.timezone || 'Europe/Warsaw';
      
      try {
        if (settings.notifications) parsedSettings.notifications = JSON.parse(settings.notifications);
        if (settings.privacy) parsedSettings.privacy = JSON.parse(settings.privacy);
        if (settings.display) parsedSettings.display = JSON.parse(settings.display);
      } catch (e) {
        console.error('❌ Error parsing user settings:', e);
      }
    }

    res.json({
      user: {
        id: user.id,
        email: user.email,
        first_name: user.first_name,
        last_name: user.last_name,
        user_type: user.user_type,
        created_at: user.created_at
      },
      profile: profile || {
        first_name: user.first_name,
        last_name: user.last_name,
        phone: null,
        date_of_birth: null,
        gender: null,
        address: null,
        city: null,
        postal_code: null,
        country: 'Poland',
        avatar_url: '/uploads/avatars/default-avatar.svg'
      },
      settings: parsedSettings
    });

  } catch (error) {
    console.error('❌ Error fetching client profile:', error);
    res.status(500).json({ 
      message: 'Error fetching profile',
      error: 'FETCH_PROFILE_ERROR'
    });
  }
});

/**
 * PUT /api/client/profile
 * Update client profile information
 */
router.put('/profile', sanitizeInput, async (req, res) => {
  try {
    const userId = req.user.id;
    const {
      first_name,
      last_name,
      phone,
      date_of_birth,
      gender,
      address,
      city,
      postal_code,
      country
    } = req.body;

    // Update basic user info
    if (first_name !== undefined || last_name !== undefined) {
      await database.run(`
        UPDATE users SET
          first_name = COALESCE(?, first_name),
          last_name = COALESCE(?, last_name),
          updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
      `, [first_name, last_name, userId]);
    }

    // Update or create extended profile
    const existingProfile = await database.get(
      'SELECT id FROM platform_clients WHERE user_id = ?',
      [userId]
    );

    if (existingProfile) {
      // Update existing profile
      await database.run(`
        UPDATE platform_clients SET
          first_name = COALESCE(?, first_name),
          last_name = COALESCE(?, last_name),
          phone = COALESCE(?, phone),
          date_of_birth = COALESCE(?, date_of_birth),
          gender = COALESCE(?, gender),
          address = COALESCE(?, address),
          city = COALESCE(?, city),
          postal_code = COALESCE(?, postal_code),
          country = COALESCE(?, country),
          updated_at = CURRENT_TIMESTAMP
        WHERE user_id = ?
      `, [first_name, last_name, phone, date_of_birth, gender, address, city, postal_code, country, userId]);
    } else {
      // Create new profile
      await database.run(`
        INSERT INTO platform_clients (
          user_id, first_name, last_name, phone, date_of_birth, gender, address, city, postal_code, country
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `, [userId, first_name, last_name, phone, date_of_birth, gender, address, city, postal_code, country]);
    }

    console.log('✅ Profile updated for user:', userId);

    // Return updated profile
    const updatedUser = await database.get(
      'SELECT id, email, first_name, last_name, created_at FROM users WHERE id = ?',
      [userId]
    );

    const updatedProfile = await database.get(
      'SELECT * FROM platform_clients WHERE user_id = ?',
      [userId]
    );

    res.json({
      message: 'Profile updated successfully',
      user: updatedUser,
      profile: updatedProfile
    });

  } catch (error) {
    console.error('❌ Error updating profile:', error);
    res.status(500).json({
      message: 'Error updating profile',
      error: 'UPDATE_PROFILE_ERROR'
    });
  }
});

// ============================================================================
// AVATAR UPLOAD
// ============================================================================

/**
 * POST /api/client/avatar
 * Upload user avatar
 */
router.post('/avatar', upload.single('avatar'), async (req, res) => {
  try {
    const userId = req.user.id;

    if (!req.file) {
      return res.status(400).json({
        message: 'No file uploaded',
        error: 'NO_FILE'
      });
    }

    const avatarUrl = `/uploads/avatars/${req.file.filename}`;

    // Update avatar URL in profile
    const existingProfile = await database.get(
      'SELECT id FROM platform_clients WHERE user_id = ?',
      [userId]
    );

    if (existingProfile) {
      await database.run(
        'UPDATE platform_clients SET avatar_url = ?, updated_at = CURRENT_TIMESTAMP WHERE user_id = ?',
        [avatarUrl, userId]
      );
    } else {
      // Create profile with avatar
      const user = await database.get('SELECT first_name, last_name FROM users WHERE id = ?', [userId]);
      await database.run(`
        INSERT INTO platform_clients (user_id, first_name, last_name, avatar_url)
        VALUES (?, ?, ?, ?)
      `, [userId, user.first_name, user.last_name, avatarUrl]);
    }

    // Save upload record
    await database.run(`
      INSERT INTO platform_uploads (user_id, file_type, filename, original_name, file_path, file_size, mime_type)
      VALUES (?, ?, ?, ?, ?, ?, ?)
    `, [userId, 'avatar', req.file.filename, req.file.originalname, req.file.path, req.file.size, req.file.mimetype]);

    console.log('✅ Avatar uploaded for user:', userId);

    res.json({
      message: 'Avatar uploaded successfully',
      avatar_url: avatarUrl,
      file_info: {
        filename: req.file.filename,
        size: req.file.size,
        mimetype: req.file.mimetype
      }
    });

  } catch (error) {
    console.error('❌ Error uploading avatar:', error);
    res.status(500).json({
      message: 'Error uploading avatar',
      error: 'UPLOAD_AVATAR_ERROR'
    });
  }
});

/**
 * DELETE /api/client/avatar
 * Remove user avatar
 */
router.delete('/avatar', async (req, res) => {
  try {
    const userId = req.user.id;

    // Get current avatar
    const profile = await database.get(
      'SELECT avatar_url FROM platform_clients WHERE user_id = ?',
      [userId]
    );

    if (profile && profile.avatar_url && !profile.avatar_url.includes('default-avatar')) {
      // Delete file from filesystem
      const filePath = path.join(__dirname, '../..', profile.avatar_url);
      if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
      }
    }

    // Update database
    await database.run(
      'UPDATE platform_clients SET avatar_url = ? WHERE user_id = ?',
      ['/uploads/avatars/default-avatar.svg', userId]
    );

    console.log('✅ Avatar removed for user:', userId);

    res.json({
      message: 'Avatar removed successfully',
      avatar_url: '/uploads/avatars/default-avatar.svg'
    });

  } catch (error) {
    console.error('❌ Error removing avatar:', error);
    res.status(500).json({
      message: 'Error removing avatar',
      error: 'REMOVE_AVATAR_ERROR'
    });
  }
});

// ============================================================================
// SETTINGS MANAGEMENT
// ============================================================================

/**
 * GET /api/client/settings
 * Get client settings
 */
router.get('/settings', async (req, res) => {
  try {
    const userId = req.user.id;

    const settings = await database.get(
      'SELECT language, timezone, notifications, privacy, display, created_at, updated_at FROM platform_user_settings WHERE user_id = ?',
      [userId]
    );

    let parsedSettings = {
      language: 'pl',
      timezone: 'Europe/Warsaw',
      notifications: {
        email_marketing: true,
        email_bookings: true,
        email_reminders: true,
        push_marketing: false,
        push_bookings: true,
        push_reminders: true
      },
      privacy: {
        profile_visibility: 'public',
        show_activity: false,
        allow_contact: true,
        data_analytics: true
      },
      display: {
        theme: 'system',
        compact_mode: false,
        animations: true
      }
    };

    if (settings) {
      parsedSettings.language = settings.language || 'pl';
      parsedSettings.timezone = settings.timezone || 'Europe/Warsaw';
      
      try {
        if (settings.notifications) parsedSettings.notifications = JSON.parse(settings.notifications);
        if (settings.privacy) parsedSettings.privacy = JSON.parse(settings.privacy);
        if (settings.display) parsedSettings.display = JSON.parse(settings.display);
      } catch (e) {
        console.error('❌ Error parsing settings:', e);
      }
    }

    res.json(parsedSettings);

  } catch (error) {
    console.error('❌ Error fetching settings:', error);
    res.status(500).json({ 
      message: 'Error fetching settings',
      error: 'FETCH_SETTINGS_ERROR'
    });
  }
});

/**
 * PUT /api/client/settings
 * Update client settings
 */
router.put('/settings', sanitizeInput, async (req, res) => {
  try {
    const userId = req.user.id;
    const { language, timezone, notifications, privacy, display } = req.body;

    // Check if settings exist
    const existingSettings = await database.get(
      'SELECT id FROM platform_user_settings WHERE user_id = ?',
      [userId]
    );

    if (existingSettings) {
      // Update existing settings
      await database.run(`
        UPDATE platform_user_settings SET
          language = COALESCE(?, language),
          timezone = COALESCE(?, timezone),
          notifications = COALESCE(?, notifications),
          privacy = COALESCE(?, privacy),
          display = COALESCE(?, display),
          updated_at = CURRENT_TIMESTAMP
        WHERE user_id = ?
      `, [
        language,
        timezone,
        notifications ? JSON.stringify(notifications) : null,
        privacy ? JSON.stringify(privacy) : null,
        display ? JSON.stringify(display) : null,
        userId
      ]);
    } else {
      // Create new settings
      await database.run(`
        INSERT INTO platform_user_settings (user_id, language, timezone, notifications, privacy, display)
        VALUES (?, ?, ?, ?, ?, ?)
      `, [
        userId,
        language || 'pl',
        timezone || 'Europe/Warsaw',
        notifications ? JSON.stringify(notifications) : null,
        privacy ? JSON.stringify(privacy) : null,
        display ? JSON.stringify(display) : null
      ]);
    }

    console.log('✅ Settings updated for user:', userId);

    res.json({
      message: 'Settings updated successfully',
      language, timezone, notifications, privacy, display
    });

  } catch (error) {
    console.error('❌ Error updating settings:', error);
    res.status(500).json({
      message: 'Error updating settings',
      error: 'UPDATE_SETTINGS_ERROR'
    });
  }
});

// ============================================================================
// PASSWORD CHANGE
// ============================================================================

/**
 * PUT /api/client/password
 * Change user password
 */
router.put('/password',
  sanitizeInput,
  validateRequiredFields(['currentPassword', 'newPassword']),
  async (req, res) => {
    try {
      const userId = req.user.id;
      const { currentPassword, newPassword } = req.body;

      // Get current password hash
      const user = await database.get(
        'SELECT password FROM users WHERE id = ?',
        [userId]
      );

      if (!user) {
        return res.status(404).json({
          message: 'User not found',
          error: 'USER_NOT_FOUND'
        });
      }

      // Verify current password
      const isCurrentPasswordValid = await bcrypt.compare(currentPassword, user.password);

      if (!isCurrentPasswordValid) {
        return res.status(400).json({
          message: 'Current password is incorrect',
          error: 'INVALID_CURRENT_PASSWORD'
        });
      }

      // Validate new password
      if (newPassword.length < 6) {
        return res.status(400).json({
          message: 'New password must be at least 6 characters long',
          error: 'PASSWORD_TOO_SHORT'
        });
      }

      // Hash new password
      const newPasswordHash = await bcrypt.hash(newPassword, config.BCRYPT_SALT_ROUNDS);

      // Update password
      await database.run(
        'UPDATE users SET password = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [newPasswordHash, userId]
      );

      console.log('✅ Password changed for user:', req.user.email);

      res.json({
        message: 'Password changed successfully'
      });

    } catch (error) {
      console.error('❌ Error changing password:', error);
      res.status(500).json({
        message: 'Error changing password',
        error: 'CHANGE_PASSWORD_ERROR'
      });
    }
  }
);

// ============================================================================
// FAQ & SUPPORT
// ============================================================================

/**
 * GET /api/client/faq
 * Get FAQ data (static for now)
 */
router.get('/faq', async (req, res) => {
  const faqData = [
    {
      id: 1,
      category: 'account',
      question: 'How do I update my profile information?',
      answer: 'You can update your profile by clicking on the "Profile" tab in your dashboard. Here you can edit your personal information, upload a profile picture, and modify your preferences.',
      tags: ['profile', 'edit', 'update', 'personal', 'information']
    },
    {
      id: 2,
      category: 'account',
      question: 'How do I change my password?',
      answer: 'To change your password, go to the "Security" tab in your dashboard. Enter your current password and choose a new, strong password. Make sure to use a combination of letters, numbers, and special characters.',
      tags: ['password', 'security', 'change', 'reset']
    },
    {
      id: 3,
      category: 'bookings',
      question: 'How do I book an appointment?',
      answer: 'To book an appointment, browse our directory of service providers, select the one you prefer, choose an available time slot, and complete the booking process. You will receive a confirmation email.',
      tags: ['booking', 'appointment', 'schedule', 'reserve']
    },
    {
      id: 4,
      category: 'bookings',
      question: 'Can I cancel or reschedule my appointment?',
      answer: 'Yes, you can cancel or reschedule your appointment up to 24 hours before the scheduled time. Go to your bookings in the dashboard and select the appointment you want to modify.',
      tags: ['cancel', 'reschedule', 'modify', 'appointment', 'booking']
    },
    {
      id: 5,
      category: 'payments',
      question: 'What payment methods do you accept?',
      answer: 'We accept all major credit cards (Visa, MasterCard, American Express), PayPal, and bank transfers. All payments are processed securely through our encrypted payment system.',
      tags: ['payment', 'credit card', 'paypal', 'bank transfer', 'methods']
    },
    {
      id: 6,
      category: 'services',
      question: 'What types of ayurvedic services are available?',
      answer: 'Our platform offers a wide range of ayurvedic services including consultations, massages, treatments, yoga sessions, meditation classes, and wellness programs. Each provider specifies their specialties.',
      tags: ['ayurveda', 'services', 'massage', 'consultation', 'yoga', 'meditation']
    },
    {
      id: 7,
      category: 'technical',
      question: 'I am having trouble logging in. What should I do?',
      answer: 'If you cannot log in, first check that you are using the correct email and password. If you forgot your password, use the "Forgot Password" link. Clear your browser cache or try a different browser if the problem persists.',
      tags: ['login', 'password', 'trouble', 'technical', 'browser', 'cache']
    },
    {
      id: 8,
      category: 'technical',
      question: 'Is my personal information secure?',
      answer: 'Yes, we take your privacy and security very seriously. All personal information is encrypted and stored securely. We never share your data with third parties without your consent, and you can control your privacy settings in your dashboard.',
      tags: ['security', 'privacy', 'personal information', 'data', 'encryption']
    }
  ];

  res.json({
    faq: faqData,
    categories: ['account', 'bookings', 'payments', 'services', 'technical']
  });
});

/**
 * POST /api/client/support
 * Submit support request
 */
router.post('/support', sanitizeInput, validateRequiredFields(['subject', 'message']), async (req, res) => {
  try {
    const userId = req.user.id;
    const { subject, message, priority = 'medium' } = req.body;

    // For now, just log the support request
    // In real implementation, save to database and send email
    console.log('📧 Support request from user:', userId, {
      subject,
      message: message.substring(0, 100) + '...',
      priority
    });

    // Generate ticket ID
    const ticketId = `AYU-${Date.now()}-${userId}`;

    res.json({
      message: 'Support request submitted successfully. We will respond within 24 hours.',
      ticket_id: ticketId,
      estimated_response: '24 hours'
    });

  } catch (error) {
    console.error('❌ Error submitting support request:', error);
    res.status(500).json({
      message: 'Error submitting support request',
      error: 'SUPPORT_REQUEST_ERROR'
    });
  }
});

module.exports = router;