const express = require('express');
const config = require('../config/config');

const router = express.Router();

// ============================================================================
// SAFE MODULE IMPORTS (with error handling)
// ============================================================================

let authRoutes, crmRoutes, clientRoutes;

try {
  authRoutes = require('./auth');
  console.log('✅ Auth routes loaded');
} catch (error) {
  console.error('❌ Error loading auth routes:', error.message);
  authRoutes = null;
}

try {
  crmRoutes = require('./crm');
  console.log('✅ CRM routes loaded');
} catch (error) {
  console.error('❌ Error loading CRM routes:', error.message);
  crmRoutes = null;
}

try {
  clientRoutes = require('./client');
  console.log('✅ Client routes loaded');
} catch (error) {
  console.error('❌ Error loading client routes:', error.message);
  clientRoutes = null;
}

// ============================================================================
// API STATUS ENDPOINT (always available)
// ============================================================================

// Root endpoint
router.get('/', (req, res) => {
  res.json({
    message: 'Ayurwell CRM API v1.0',
    status: 'active',
    timestamp: new Date().toISOString()
  });
});

// API endpoint (DODAJ TUTAJ)
router.get('/api', (req, res) => {
  res.json({
    message: 'Ayurwell API',
    status: 'active'
  });
});


/**
 * GET /api/status
 * Detailed API status endpoint
 */
router.get('/api/status', (req, res) => {
  const status = {
    api: {
      name: config.APP_NAME + ' API',
      version: config.APP_VERSION,
      status: 'operational',
      environment: config.NODE_ENV
    },
    server: {
      uptime: process.uptime(),
      memory: process.memoryUsage(),
      timestamp: new Date().toISOString(),
      port: config.PORT
    },
    modules: {
      auth_routes: authRoutes ? 'loaded' : 'not_loaded',
      crm_routes: crmRoutes ? 'loaded' : 'not_loaded',
      client_routes: clientRoutes ? 'loaded' : 'not_loaded'
    },
    features: config.FEATURES,
    database: {
      type: 'SQLite',
      path: config.DB_PATH
    }
  };

  // Add request information if in development
  if (config.IS_DEVELOPMENT) {
    status.request_info = {
      method: req.method,
      url: req.url,
      headers: req.headers,
      user_agent: req.get('User-Agent'),
      ip: req.ip
    };
  }

  res.json(status);
});

/**
 * GET /api/info
 * API information and available endpoints
 */
router.get('/api/info', (req, res) => {
  const endpoints = {
    health: [
      'GET /health - Health check',
      'GET /api/status - Detailed status',
      'GET /api/info - API information'
    ]
  };

  if (authRoutes) {
    endpoints.authentication = [
      'POST /api/register - Register new user',
      'POST /api//login - User login',
      'GET /api//verify-token - Verify JWT token',
      'POST /api//logout - User logout'
    ];
  }

  if (crmRoutes) {
    endpoints.crm = [
      'GET /api/crm/leads - Get all leads',
      'GET /api/crm/leads/:id - Get single lead',
      'POST /api/crm/leads - Create new lead',
      'PUT /api/crm/leads/:id - Update lead',
      'DELETE /api/crm/leads/:id - Delete lead',
      'GET /api/crm/activities - Get all activities',
      'POST /api/crm/activities - Create new activity',
      'PUT /api/crm/activities/:id/complete - Mark activity as completed',
      'GET /api/crm/sales-team - Get sales team',
      'POST /api/crm/sales-team - Add team member',
      'DELETE /api/crm/sales-team/:id - Remove team member',
      'GET /api/crm/users - Get all CRM users'
    ];
  }

  if (clientRoutes) {
    endpoints.client_dashboard = [
      'GET /api/client/profile - Get client profile',
      'PUT /api/client/profile - Update client profile',
      'GET /api/client/settings - Get client settings',
      'PUT /api/client/settings - Update client settings',
      'PUT /api/client/password - Change password',
      'GET /api/client/dashboard - Get dashboard info',
      'GET /api/client/uploads - Get client uploads'
    ];
  }

  res.json({
    api: {
      name: config.APP_NAME + ' API',
      version: config.APP_VERSION,
      description: 'RESTful API for CRM and Client Dashboard',
      documentation: 'Available endpoints listed below'
    },
    server: {
      timestamp: new Date().toISOString(),
      environment: config.NODE_ENV
    },
    endpoints: endpoints
  });
});

// ============================================================================
// ROUTE MODULES MOUNTING
// ============================================================================

// Authentication routes (no /api prefix - mounted directly)
if (authRoutes) {
  router.use('/api', authRoutes);
  console.log('✅ Auth routes mounted on /api');
} else {
  console.log('⚠️ Auth routes not available');
  
  // Fallback auth endpoints
  router.post('/register', (req, res) => {
    res.status(503).json({
      message: 'Authentication module not loaded',
      error: 'AUTH_MODULE_NOT_LOADED'
    });
  });
  
  router.post('/login', (req, res) => {
    res.status(503).json({
      message: 'Authentication module not loaded',
      error: 'AUTH_MODULE_NOT_LOADED'
    });
  });
  
  router.get('/verify-token', (req, res) => {
    res.status(503).json({
      message: 'Authentication module not loaded',
      error: 'AUTH_MODULE_NOT_LOADED'
    });
  });
}

// CRM routes (with /api/crm prefix)
if (crmRoutes) {
  router.use('/api/crm', crmRoutes);
  console.log('✅ CRM routes mounted on /api/crm');
} else {
  console.log('⚠️ CRM routes not available');
  
  // Fallback CRM endpoints
  router.use('/api/crm/*', (req, res) => {
    res.status(503).json({
      message: 'CRM module not loaded',
      error: 'CRM_MODULE_NOT_LOADED'
    });
  });
}

// Client Dashboard routes (with /api/client prefix)
if (clientRoutes) {
  router.use('/api/client', clientRoutes);
  console.log('✅ Client routes mounted on /api/client');
} else {
  console.log('⚠️ Client routes not available');
  
  // Fallback client endpoints
  router.use('/api/client/*', (req, res) => {
    res.status(503).json({
      message: 'Client dashboard module not loaded',
      error: 'CLIENT_MODULE_NOT_LOADED'
    });
  });
}

// ============================================================================
// LOGGING MIDDLEWARE (development only)
// ============================================================================

if (config.IS_DEVELOPMENT) {
  router.use('*', (req, res, next) => {
    console.log(`📡 ${req.method} ${req.originalUrl}`, {
      timestamp: new Date().toISOString(),
      ip: req.ip,
      user_agent: req.get('User-Agent')?.substring(0, 50) + '...'
    });
    next();
  });
}

// ============================================================================
// ERROR HANDLING FOR ROUTES
// ============================================================================

// Handle 404 for API routes
router.use('/api/*', (req, res) => {
  res.status(404).json({
    message: 'API endpoint not found',
    error: 'ENDPOINT_NOT_FOUND',
    requested_path: req.path,
    available_endpoints: '/api/info'
  });
});

// Global error handler for routes
router.use((error, req, res, next) => {
  console.error('🚨 Route Error:', error);
  
  res.status(500).json({
    message: 'Internal server error',
    error: 'INTERNAL_ERROR',
    ...(config.IS_DEVELOPMENT && { 
      details: error.message,
      stack: error.stack 
    })
  });
});

// ============================================================================
// EXPORTS
// ============================================================================

module.exports = router;